%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 2 - Section 3:  DFT vs DTFT and IDFT
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger                          Date: May 2013   

clc;
clear all;
close all;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Unpublished Code of (DTFT)

N           = 15;                     %% Sequence Length
n           = 0:1:N-1;
Rectangular = ones(1,N);              %% Rectangular Window definition

Fs          = 1e6;          %% Hz
Ts          = 1/Fs;         %% Sample Period
fo		    = 100e3;        %% Hz  
TestSequence= exp(j*2*pi*n*Ts*fo); 
    
FResolution  = 0.001;                 %% DFT Resolution in Hertz 
Frequencies  = -0.5:FResolution:0.5;  %% Test Frequencies
DTFT_Out     = zeros(1,length(Frequencies));
     
for i = 1:length(Frequencies)         %% DTFT Calculation
    f            = Frequencies(1,i);  %% Current Analysis Frequency
    AnalysisTone = exp(j*2*pi*n*f);
    DTFT_Out(1,i)= (1/N)*(TestSequence.*Rectangular)*AnalysisTone';
end
MagSpectrum1 = abs(DTFT_Out);
    


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (page 127)

%% DFT Calculation
N           = 15;           %% Sequence Length
n           = 0:1:N-1;      %% Time Index
m           = 0:1:N-1;      %% Frequency Index
    
Fs          = 1e6;          %% Sample Rate in Hz
Ts          = 1/Fs;         %% Sample Period
fo		    = 100e3;        %% Frequency of Complex Test Tone in Hz  

TestSequence= exp(j*2*pi*n*Ts*fo); 
FrequenciesDFT = m/N;       %% Analysis frequencies
DFT_Out        = zeros(1,N);

for i = 1:N                 %% DFT Calculation
    f            = FrequenciesDFT(1,i);
    AnalysisTone = exp(j*2*pi*n*f);
    DFT_Out(1,i) = (1/N)*(TestSequence.*Rectangular)*AnalysisTone';
end
MagSpectrumDFT   = abs(DFT_Out);
PhaseSpectrumDFT = angle(DFT_Out); 
    


%% Correct aliased frequencies that are > 0.5Hz by subtracting - 1
FrequenciesDFT(1,ceil(N/2):end) = FrequenciesDFT(1,ceil(N/2):end) - 1;
    
figure(4);
plot(n, real(TestSequence), 'k-o'); hold on; 
plot(n, imag(TestSequence), 'k--o','LineWidth', 1.2);
legend('real','imag');
title('Test Sequence'); xlabel('n');
    
figure(5);
plot(Frequencies*Fs,(MagSpectrum1), 'k--', 'LineWidth', 1.5); hold on; 
title('DTFT and DFT Magnitude of x[n]');xlabel('Frequency in Hertz');
stem(FrequenciesDFT*Fs, MagSpectrumDFT,'ko');
legend('DTFT Result','DFT Result');



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Page 129)

%% IDFT Calculation
N           = 15;           %% Sequence Length
n           = 0:1:N-1;      %% Time Index
    
IDFT_Out = zeros(1,length(n));
for i = 1:N                           %% IDFT Calculation
    m             = i - 1;
    f             = m/N;
    AnalysisTone  = exp(j*2*pi*n*f);
    IDFT_Out      = IDFT_Out + DFT_Out(1,i)*AnalysisTone;
end


%  IDFT_Out
figure(6);
plot(n, real(IDFT_Out), 'r-o'); grid on; title('IDFT Result'); hold on;
plot(n, imag(IDFT_Out), 'b-o');