%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 2 - Section 3:  Radix-4 N=64 FFT
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger                          Date: May 2013


clc;
clear all;
close all;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Page 146)

Memory     = zeros(64, 4);
Length     = 64;

%% Input Sequence cos(2 pi Freq *n/SampleRate)
n              = 0:1:Length-1;
Freq           = .05;
SampleRate     = 1;
Input          = cos(2*pi*Freq*n/SampleRate);

Order  = 1 + [0 16 32 48   4 20 36 52   8 24 40 56   12 28 44 60 ...
              1 17 33 49   5 21 37 53   9 25 41 57   13 29 45 61 ...
              2 18 34 50   6 22 38 54  10 26 42 58   14 30 46 62 ...
              3 19 35 51   7 23 39 55  11 27 43 59   15 31 47 63];
Memory(Order,1) = Input.'; 

for Section = 1:1:3
    N     = 4^Section;     %% Current section contains N-point DFT Modules
    Reps  = 64/N;          %% Number of N-point DFT Modules in current section
    I     = 4^(Section-1); %% Limit for index i
    
    Offset  = 1;
    for a = 0:Reps-1
        for i = 0:I-1      %% Start of Radix-4 Butterfly
            Ini      = Memory( Offset + i        , Section);
            IniN_4   = Memory( Offset + i + N/4  , Section);
            IniN_2   = Memory( Offset + i + N/2  , Section);
            Ini3N_4  = Memory( Offset + i + 3*N/4, Section);
            
            Wi       = exp(-j*2*pi*i/N);
            WiN_4    = exp(-j*2*pi*(i+N/4)/N);
            WiN_2    = exp(-j*2*pi*(i+N/2)/N);
            Wi3N_4   = exp(-j*2*pi*(i+3*N/4)/N);
            
            W2i      = exp(-j*2*pi*2*i/N);
            W2iN_4   = exp(-j*2*pi*2*(i+N/4)/N);
            W2iN_2   = exp(-j*2*pi*2*(i+N/2)/N);
            W2i3N_4  = exp(-j*2*pi*2*(i+3*N/4)/N);
                            
            W3i      = exp(-j*2*pi*3*i/N);
            W3iN_4   = exp(-j*2*pi*3*(i+N/4)/N);
            W3iN_2   = exp(-j*2*pi*3*(i+N/2)/N);
            W3i3N_4  = exp(-j*2*pi*3*(i+3*N/4)/N);
            
            Memory( Offset + i        , Section + 1) = ...
                            Ini + Wi*IniN_4     + W2i*IniN_2     + W3i*Ini3N_4;
            Memory( Offset + i + N/4  , Section + 1) = ...
                            Ini + WiN_4*IniN_4  + W2iN_4*IniN_2  + W3iN_4 *Ini3N_4;
            Memory( Offset + i + N/2  , Section + 1) = ...
                            Ini + WiN_2*IniN_4  + W2iN_2*IniN_2  + W3iN_2 *Ini3N_4;
            Memory( Offset + i + 3*N/4, Section + 1) = ...
                            Ini + Wi3N_4*IniN_4 + W2i3N_4*IniN_2 + W3i3N_4*Ini3N_4;
        end                %% End of Radix-4 Butterfly
    Offset = Offset + N;
    end
end
Output      = Memory(:, 4).'; 

Out_Perfect = fft(Input);  %% FFT as computed by native matlab function

Out_Mine    = Memory(:,4); %% FFT as computed by code above

[Out_Perfect.' Memory(:,4)] %% The two compared

figure(1);
subplot(3,1,1);
plot(real(Input), 'k'); grid on; hold on;
plot(imag(Input), 'k:'); title('Time Domain Input Signal');
xlabel('Discrete Time Index n');

subplot(3,1,2);
stem((0:63), real(Output), 'k'); grid on; title('Real Portion of 64-Point FFT Result');
xlabel('Discrete Frequency Index m'); axis([0 63 -30 30]);

subplot(3,1,3);
stem((0:63), imag(Output), 'k'); grid on; title('Imaginary Portion of 64-Point FFT Result');
xlabel('Discrete Frequency Index m'); axis([0 63 -30 30]);