%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 2 - Section 4:  Frequency Sampling Method of FIR design
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger                          Date: May 2013

%% FIR Filter Design 1

clc;
clear all;
close all;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Page 154)

%% Step 1: Defining the Frequency Response
N                 = 13;                   %% Sequence Length
n                 = (0:N-1) - floor(N/2); %% Time index at which to evaluate
                                          %% the IDFT
MagResponse       = [1 1 1 0 0 0 0 0 0 0 0 1 1];
PhaseResponse     = [0 0 0 0 0 0 0 0 0 0 0 0 0];
H                 = MagResponse.*cos(PhaseResponse) + ...
                  j*MagResponse.*sin(PhaseResponse);
              
%% Step 2: Taking the inverse discrete Fourier Transform
h = zeros(1, N);                          %% Impulse response h[n]
for m = 0:N-1                             %% IDFT Calculation
    f             = m/N;                  %% Normalized Frequencies
    h             = h + H(1,m+1)*exp(j*2*pi*n*f);
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Page 155)

%% Step 2b: Normalize h[n] and compute the frequency response H(f)
h_norm            = (1/abs(sum(h)))*h;    %% Normalizing h[n]

Frequencies  = -0.5:.002:0.5;
DTFT_Output  = zeros(1,length(Frequencies));

for i=1:length(Frequencies)
    f                = Frequencies(1,i);
    AnalysisTone     = exp(j*2*pi*n*f);
    DTFT_Output(1,i) = h_norm*AnalysisTone';
end
MagnitudeResponse    = abs(DTFT_Output);
MagnitudeResponse_dB = 20*log10(abs(DTFT_Output));

figure(1);
subplot(1,2,1);
plot(Frequencies, MagnitudeResponse, 'k', 'LineWidth', 1); grid on; hold on;
stem([(1/N*(0:floor(N/2)))  (1/N)*((floor(N/2)+1):(N-1))-1], MagResponse, 'k', 'LineWidth', 1);
axis([-.5 .5 0 1.2]); title('Final Magnitude Response of Filter');
xlabel('Normalized Frequency'); 
legend('Actual Response','Specified Response');
set(gca, 'XTick', [-.4 -.2 0 .2 .4]);
set(gca, 'YTick', [0 0.5 1.0]);
subplot(1,2,2);
plot(Frequencies, MagnitudeResponse_dB, 'k', 'LineWidth', 1); grid on; hold on;
xi = [0.15 0.175 .20];
yi = interp1(Frequencies, MagnitudeResponse_dB, xi, 'spline');

plot(xi, yi, 'ko', 'LineWidth', 1);
axis([-.5 .5 -60 5]); title('Final Magnitude Response of Filter in dB');
xlabel('Normalized Frequency'); ylabel('dB');
set(gca, 'XTick', [-.4 -.2 0 .2 .4]);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Page 157) b = 0.2 rather than b = 1

%% Step 3: Adding a custom Hanning Window
n         = 0:1:N-1;
b         = 0.2;                %% Exponent of customizabled Hanning window
CustHann  = (0.5 - 0.5*cos(2*pi*(n+1)/(N+1))).^b;
New_h     = (1/abs(sum(h.*CustHann)))*h.*CustHann; %% Windowed Impulse Response


%% Step 4: Computing the new Frequency Response via the DTFT
Frequencies  = -0.5:.002:0.5;
DTFT_Output  = zeros(1,length(Frequencies));

for i=1:length(Frequencies)
    f                = Frequencies(1,i);
    AnalysisTone     = exp(j*2*pi*n*f);
    DTFT_Output(1,i) = New_h*AnalysisTone';
end
MagnitudeResponse    = abs(DTFT_Output);
MagnitudeResponse_dB = 20*log10(abs(DTFT_Output));

figure(2);
subplot(2,2,1);
stem((0:N-1) - floor(N/2), New_h,'k', 'LineWidth', 1); grid on; hold on;
plot((0:N-1) - floor(N/2), New_h,'k--', 'LineWidth', 1); title('Windowed Impulse Response (b=0.2)');
axis([-N/2 N/2 -0.1 .5]);
set(gca, 'YTick', [0 .2 .4 ]);
subplot(2,2,3);
plot(Frequencies, MagnitudeResponse_dB, 'k', 'LineWidth', 1); grid on;
axis([-.5 .5 -60 5]); title('Final Magnitude Response of Filter (b=0.2)');
xlabel('Normalized Frequency'); ylabel('dB');
set(gca, 'XTick', [-.4 -.2 0 .2 .4]);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Page 157) b = 1

%% Step 3: Adding a custom Hanning Window
n         = 0:1:N-1;
b         = 1;                %% Exponent of customizabled Hanning window
CustHann  = (0.5 - 0.5*cos(2*pi*(n+1)/(N+1))).^b;
New_h     = (1/abs(sum(h.*CustHann)))*h.*CustHann; %% Windowed Impulse Response


%% Step 4: Computing the new Frequency Response via the DTFT
Frequencies  = -0.5:.002:0.5;
DTFT_Output  = zeros(1,length(Frequencies));

for i=1:length(Frequencies)
    f                = Frequencies(1,i);
    AnalysisTone     = exp(j*2*pi*n*f);
    DTFT_Output(1,i) = New_h*AnalysisTone';
end
MagnitudeResponse    = abs(DTFT_Output);
MagnitudeResponse_dB = 20*log10(abs(DTFT_Output));



subplot(2,2,2);
stem((0:N-1) - floor(N/2), New_h,'k', 'LineWidth', 1); grid on; hold on;
plot((0:N-1) - floor(N/2), New_h,'k--', 'LineWidth', 1); title('Windowed Impulse Response (b=1)');
axis([-N/2 N/2 -0.1 .5]); set(gca, 'YTick', [0 .2 .4 ]);
subplot(2,2,4);
plot(Frequencies, MagnitudeResponse_dB, 'k', 'LineWidth', 1); grid on;
axis([-.5 .5 -60 5]); title('Final Magnitude Response of Filter (b=1)');
xlabel('Normalized Frequency'); ylabel('dB');
set(gca, 'XTick', [-.4 -.2 0 .2 .4]);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Page 158)

%% Defining the Composite Input Sequence
n            = 0:200;
TestWaveform = cos(2*pi*n*1/13) + cos(2*pi*n*4/13);
Output       = zeros(1,length(TestWaveform));
DelayLine    = zeros(1,13);
h            = [0.0032 -0.0037 -0.0363 -0.0331  0.0895 0.2871  0.3866  ...
                0.2871  0.0895 -0.0331 -0.0363 -0.0037  0.0032];

%% The Convolution Operation
for n = 1:length(TestWaveform)
    DelayLine(1,2:13) = DelayLine(1,1:12);  %% Time Advance
    DelayLine(1,1)    = TestWaveform(1,n);  %% New sample enters
    Output(1,n)       = DelayLine*h.';  %% Calculating the Output
end

n1            = -1:.2:200;
TestWaveform1 = cos(2*pi*n1*1/13) + cos(2*pi*n1*4/13);

figure(3);
subplot(2,1,1);
stem(0:200, TestWaveform, 'ko', 'LineWidth', 1);  hold on; title('Test Waveform');
plot(n1, TestWaveform1, 'k--', 'LineWidth', 1); xlabel('Discrete Time - n');
axis([1 50 -2 2]);
subplot(2,1,2);
plot(real(Output), 'k--', 'LineWidth', 1); title('Output Waveform'); hold on;
stem(real(Output), 'k', 'LineWidth', 1);  xlabel('Discrete Time - n');
axis([1 50 -2 2]);