%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 2 - Section 3:  Spectrum Function
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger                          Date: May 2013

%  This function will produce an average Magnitude Specturm that is 
%  similar in look and feel to what is seen on the spectrum analyzer

%  Input Arguments
%  1. Input  --> Is the input waveform
%  2. N      --> The number of desired points for the DFT operation
%  3. SampleRate --> Only useful for the return vector f

%  Output Arguments
%  1. Mag    --> Averaged Magnitude Spectrum in dB
%  2. f      --> Frequency vector associated with Mag

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Page 130)

function [Mag f] = Spectrum(Input, N, SampleRate)

if(N < 2)
    msgbox('N must be larger than 1','Error Message');
    Mag = []; f = [];
    return;
end

if(length(Input) < N)
    msgbox('Length of Input must be larger than N','Error Message');
    Mag = []; f = [];
    return;
end

Step  = floor((length(Input)- N) / 200);
if(Step < 1)
    msgbox('Please increase Input vector length','Error Message');
    Mag = []; f = [];
    return;
end

%% Setup parameters for the DFT calculation
n          = 0:N-1;                          %% Discrete Time Index
Hanning    = 0.5 - 0.5*cos(2*pi*(n+1)/(N+1));%% Basic Hanning Definintion
H          = (N/sum(Hanning)) * Hanning;     %% Normalized Hanning Window

DFTOutput  = zeros(1,N);                     %% X[m]

%% Here we calculate average the magnitude response 
for Repetitions = 0:199
    Current_Range = (Repetitions*Step + 1):(Repetitions*Step + N);
    x             = Input(1,Current_Range);
    for m = 0:N-1                             %% The DTFT Function
        DFTOutput(1,m+1) = DFTOutput(1,m+1) + ...
                          (1/N)*abs( (x.*H) * exp(-j*2*pi*n*m/N).' );
    end
end

%% Move the aliased region above f >= 0.5 to the negative frequencies
if(mod(N,2) == 0)  %% even number
    NegStart = (N/2) + 1;
    PosStop  = (N/2);
else
    NegStart = ceil(N/2);
    PosStop  = floor(N/2);
end

DFTOutput           = [ DFTOutput(1, NegStart:end) DFTOutput(1,1:PosStop) ];
F_Norm     = 0:1/N:(N-1)/N;                   %% Normalized Frequency
                                 
%% Function Outputs
Mag        = 20*log10(DFTOutput/200);
f          = [ F_Norm(1, NegStart:end)-1  F_Norm(1,1:PosStop) ]*SampleRate; 
                                         %% Actual reordered frequencies

