%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 3 - Section 3:  DFT/IDFT based Interpolation
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger                          Date: May 2013

clc;
clear all;
close all;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Page 207 and 208)

%% Input waveform Definition (Sample Rate is 1Hz)
N     = 17;          %% Length N of input sequence x[n]
n     = 0:1:N-1;     %% normal discrete time index n
x_n   = cos((2*pi*n*2/N)+pi/8) +     sin(2*pi*n*4/N) + ... 
     0.5*cos((2*pi*n*.2/N))     -  n/16;   %% x[n] = Input Waveform
 
t     = 0:.1:24;     %% high resolution time index
x_t   = cos((2*pi*t*2/N)+pi/8) +     sin(2*pi*t*4/N) + ... 
     0.5*cos((2*pi*t*.2/N))     - .5* t/8;   %% x(t) = high res verstion

 
%% Computing the DFT of x[n]
m           = 0:1:N-1;     %% Frequency index m
UseCorrect  = 1;           %% 0/1 = aliased/correct norm. frequencies
dFreq       = zeros(1,N);  %% holds normalized frequencies
for i = 1:N
    if(m(1,i)/N >= 0.5 && UseCorrect == 1); dFreq(1,i) = m(1,i)/N -1;
    else                                    dFreq(1,i) = m(1,i)/N;
    end
end

DFT_Out     = zeros(1,N);
for i = 1:N                 %% DFT Calculation
    f            = dFreq(1,i);
    AnalysisTone = exp(j*2*pi*n*f);
    DFT_Out(1,i) = (1/N)*x_n*AnalysisTone';
end


%% Interpolation via the IFT at any time instance 0 <= t < N
IDFT_Out = zeros(1,length(t));
for i = 1:N                 %% IDFT Calculation
    f             = dFreq(1,i);
    AnalysisTone  = exp(j*2*pi*t*f);
    IDFT_Out      = IDFT_Out + DFT_Out(1,i)*AnalysisTone;
end

figure(1);
plot(t, real(IDFT_Out), 'k', 'LineWidth',0.8); hold on;
plot(t, x_t, 'k--', 'LineWidth',0.8);
plot(n, x_n, 'ko', 'LineWidth',0.8);
legend('Interpolator Result', 'Original High Resolution Waveform', ...
       'Input Samples');
title('Results'); xlabel('Time in seconds');
axis([0 24 -3 3]);
set(gca,  'LineWidth', 1);


return
figure(1);
subplot(2,1,1);
plot(n, real(x_n), 'ko'); grid on; title('Time Sequence'); hold on;
xlabel('discrete time n');
subplot(2,1,2);
stem(dFreq, abs(DFT_Out), 'k'); grid on; hold on; title('DFT Magnitude');
xlabel('normalized frequency');

                     

