%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 3 - Section 3:  FIR Filter based Interpolation
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger                          Date: May 2013

clc;
clear all;
close all;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Pages 202)

N     = 13;
m     = [-6 -5 -4 -3 -2 -1 0 1 2 3 4 5 6];     %% Force Response via the
Mag   = [1 1 1 1 1 1 1 1 1 1 1 1 1];           %% Frequency Sampling Method
no    = 6.75;                                  %% Sample Delay
Phase = -2*pi*m*no/N;
H     = Mag.*cos(Phase) + j*Mag.*sin(Phase);   %% Rectangular Freq Response




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Pages 205)

%% Computing the inverse discrete Fourier transform
DiscreteTime = 0:N-1;       
h = zeros(1,N);
for i=1:N
    n      = DiscreteTime(1,i);
    h(1,i) = H*exp(j*2*pi*m*n/N).';
end

%% Hanning Overlay
n                  = 0:N-1;          
Hanning            = (ones(1,N) - cos(2*pi*(n+1)/(N+1))); 
h_windowed         = h.*Hanning/sum(Hanning.*h); % Windowed impulse response
                                                 % scaled for DC gain = 1.0

%% Calculated frequency response via DTFT  done separately
DiscreteFrequency  = -6.5:.1:0; %% Negative Fractionally Valued Frequencies
NegResult          = zeros(1,length(DiscreteFrequency));
for i=1:length(DiscreteFrequency)
    m              = DiscreteFrequency(1,i);
    NegResult(1,i) = h_windowed*exp(-j*2*pi*n*m/N).';
end
 
DiscreteFrequency  = 0.1:.1:6.5; %% Positive fractionally valued frequencies
PosResult          = zeros(1,length(DiscreteFrequency));
for i=1:length(DiscreteFrequency)
    m              = DiscreteFrequency(1,i);
    PosResult(1,i) = h_windowed*exp(-j*2*pi*n*m/N).';
end
 
%% Required operation to properly unwrap the phase response for group delay %% calculation
MagNew            = abs([NegResult PosResult]);
PhaseNew          = [(fliplr(unwrap(fliplr(angle(NegResult))))) ...
                       unwrap(angle(PosResult))]; 
DiscreteFrequency = -6.5:.1:6.5;
 
%% Calculate group delay
GroupDelay = zeros(1,length(PhaseNew));
for i = 2:length(PhaseNew) - 1
    FrequencyDelta = (DiscreteFrequency(1, i+1)-DiscreteFrequency(1, i))/N;
    PhaseResponseDelta = PhaseNew(1, i+1) - PhaseNew(1, i);
    GroupDelay(1,i)    = (-1/(2*pi))*PhaseResponseDelta/FrequencyDelta;
end
 
GroupDelay(1,1) =  GroupDelay(1,2);
GroupDelay(1,length(PhaseNew)) =  GroupDelay(1,length(PhaseNew)-1);


figure(3); 
subplot(3,1,1);
plot(DiscreteFrequency, MagNew,'k'); grid on; hold on;
stem(-6:6, Mag,'k'); xlabel('m');
title('Magnitude Response of Windowed FIR Filter'); axis([-6.5 6.5 0 1.2]);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Pages 206)

subplot(3,1,2);
plot(-6:6, Phase,'ko'); grid on; hold on;
plot(DiscreteFrequency, PhaseNew,'k'); 
title('Phase Response of Windowed FIR Filter'); ylabel('radians'); xlabel('m');
axis([-6.5 6.5 -20 20]);
subplot(3,1,3);
plot(DiscreteFrequency, GroupDelay,'k'); grid on; hold on; 
title('Group Delay of Windowed FIR Filter'); ylabel('Samples'); xlabel('m');
axis([-6.5 6.5 6 8]);	
