%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 3 - Section 2: Remez Exchange Function
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger                          Date: May 2013

%%  This function finds the new points at which we will recalculate
%%  the polynomial coefficients for the minmax criterion polyfit

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%5
%% Code (Page 190)


function [Output] = Remez(ErrorFunction, Order, PreviousPointsIndex)

%% ErrorFunction --> Is the function passed for evaluation
%% Order --> This is the order for the minmax fit. This number is only
%%           used for error checking.
%% PreviousPointsIndex --> Index of Previous Try

%% The return value is the index at which the new max values were found

NumberOfRoots  = Order + 1;
NumberOfPoints = Order + 2;

%% Finding roots (via brute search) / Stores segment boundaries in Range

Range   = zeros(NumberOfPoints, 2); 
Counter = 1;
for r = 1:NumberOfRoots
    Range(r,1) = Counter;
    CurrentSign = sign(ErrorFunction(Counter));
    while(sign(ErrorFunction(Counter)) == CurrentSign  ...
                                   && Counter < length(ErrorFunction))
        Counter = Counter + 1;
    end
    Range(r,2) = Counter - 1;     %% Change in sign was found
end
Range(NumberOfPoints,1) = Range(NumberOfPoints-1,2) + 1;
Range(NumberOfPoints,2) = length(ErrorFunction);

if( Range(NumberOfPoints-1,1) == length(ErrorFunction) )
           %% Houston, we have a problem, we don't have enough roots
    Output = PreviousPointsIndex;
    return;
end

%% Let's now look through each segment to find the min max values

StartIndex = 0;
for r=1:NumberOfPoints
    CurrentSection = ErrorFunction(Range(r,1):Range(r,2));
    [Cmax Imax] = max(CurrentSection);
    [Cmin Imin] = min(CurrentSection);
    
    if(Cmax > abs(Cmin))
        Output(1,r) = Imax + StartIndex;
    else
        Output(1,r) = Imin + StartIndex;
    end
    StartIndex = Range(r,2);
end
