%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 4 - Section 3:  MMSE Equalizer Example
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger                          Date: May 2013

clc;
clear all;
close all;

%% 1. Definitions
VarIn   = 1;                         %% Variance of input signal
VarN    = 1e-4;                      %% Noise Variance - normal 1e-4 /
%VarN    = 5e-2;
Channel = [.1 -.1 .05 .9+j*.1 .05];  %% L = 5
%Channel = [.01 -.01 .3 .9+j*0.1 .05];
N       = 5;                         %% Number of taps in equalizer
%N      = 12;
L       = length(Channel);           %% Number of taps in channel
D       = N + L - 4;                 %% D = 6 / Delay must be less than N + L

%% 2. Calculate H
row     = zeros(1,N+L-1); row(1,1:L) = Channel;
col     = zeros(1,N);     col(1,1)   = Channel(1,1);
H       = toeplitz(col,row);

%% 3. Calculate Ry
RIn     = VarIn*eye(N+L-1);
Rv      = VarN*eye(N);
Ry      = H*RIn*H' + Rv;

%% 4. Calculate H E(In(n-(N-1))IN(n)  
E_InIN            = zeros(1,N+L-1);
E_InIN(1,1+D)     = VarIn;             %% 1+D due to MatLab array indexing 
E_InIN_H          = E_InIN*H';
 
EC                = E_InIN_H*inv(Ry);  %% Row Vector C
              
Corrected_Channel = conv(Channel, EC); %% The corrected channel

%% 5. Generate input sequence and filter it through the channel
SimL   = 2000;                      %% Number of Symbols in Simulation
Input  = sqrt(VarIn)*(sign(randn(1,SimL)) + ...  
					i*sign(randn(1,SimL)))/sqrt(2);
                                    %% QPSK input sequence (variance=VarS)
y      = filter(Channel,1,Input);   %% Variance has changed after filter
Noise  = sqrt(VarN)*(((randn(1,SimL)) + i*(randn(1,SimL))))/sqrt(2);
                                    %% Complex Noise (variance VarN)
Observations = y + Noise;

%% 6. The Equalization and Analysis Process
Estimation       = filter(EC,1,Observations); %% Executing the equalizer
EstimationVector = Estimation(1,D+1:SimL);    %% Estimated Samples
InputVector      = Input(1,1:SimL-D);         %% Reference Samples

Error            = EstimationVector - InputVector;
EVM_Percent_RMS  = sqrt(var(Error)/var(InputVector))*100
EVM_dB           = 10*log10(var(Error)/var(InputVector))
SNR              = 10*log10(var(y)/VarN)


figure(1);
subplot(2,2,1);
plot(real(Observations(1,10:end)), imag(Observations(1,10:end)), 'k.'); grid on; title('Observation Y[n]');
set(gca, 'XTick', [-.707 0 .707]);
set(gca, 'YTick', [-.707 0 .707]);
axis([-1 1 -1 1]); set(gca, 'LineWidth', 1.1);
subplot(2,2,2);
plot(real(EstimationVector), imag(EstimationVector), 'k.'); grid on;
title(['Estimate of I[n] with EVM = ' num2str(EVM_dB) 'dB']);
set(gca, 'XTick', [-.707 0 .707]);
set(gca, 'YTick', [-.707 0 .707]); set(gca, 'LineWidth', 1.1);
axis([-1 1 -1 1]);

subplot(2,2,3);
stem(abs(Corrected_Channel), 'k', 'LineWidth', 0.9); grid on; title('Corrected Channel'); set(gca, 'LineWidth', 1.1);

[h,f]  = freqz(Channel,1,64,1);Mag = 20*log10(abs(h));Mag = Mag - max(Mag);
[h2,f] = freqz(Corrected_Channel,1,64,1);Mag2 = 20*log10(abs(h2));Mag2 = Mag2 - max(Mag2);

subplot(2,2,4)
plot(f,Mag,'k--', 'LineWidth', 1.0); grid off; hold on;
plot(f,Mag2,'k', 'LineWidth', 1.0);
title('Magnitude Responses');
xlabel('Normalized Frequency'); ylabel('dB');
legend('Original Channel','Equalized Channel');
axis([0 .5 -5 1]);
set(gca, 'XTick', [.1 .2 .3 .4]);  
set(gca, 'YTick', [-4 -2 0]);

figure(2);
subplot(1,2,1);
stem(real(Channel), 'k'); grid on; hold on; title('Channel Impulse Response');
stem(imag(Channel), 'kx'); legend('Real','Imag');
subplot(1,2,2);
stem(real(Corrected_Channel), 'k'); grid on; hold on; title('Equalized Channel Impulse Response');
stem(imag(Corrected_Channel), 'kx'); legend('Real','Imag');
axis([1 9 -.5 1]);
