%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 4 - Section 4:  The LMS Technique
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger                          Date: May 2013

clc;
clear all;
close all;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Page 241-242)

%% 1. Defining the complex input stream and Channel Coefficients
 
InputLength = 1500;
In          = sign(randn(1,InputLength)) + sign(randn(1,InputLength))*j;
Channel     = [-.19j  .14  1+.1j  -.16  +.11j + .03];
 
y           = filter(Channel,1, In);   %% The Observation

figure(1);
subplot(2,1,1); 
stem(real(In(1,1:100)), 'k'); grid on; axis([1 80 -1.2 1.2]);
title('Real Part of In[n]');
subplot(2,1,2); 
stem(real(y(1,1:100)),  'k'); grid on; axis([1 80 -2 2]);
title('Real Part of the observation y[n]');


%% 2. Initialize the Estimator to be an All-pass Filter
 
EC  =  [0 0 0 0 0 0 1 0 0 0 0 0 0 ]; %% Initialize equalizer EC=C' 
                                     %% as simple all pass 
[M Delay_Channel]   = max(Channel);  %% Get Approximate Channel Delay
[M Delay_Estimator] = max(EC);       %% Get Initial Equalizer Delay
Approximate_Delay   = Delay_Channel + Delay_Estimator;
N = length(EC);                      %% Number of taps in the estimator
Y = zeros(N,1);              %% Y -> content of estimator shift register
                                     %% Y = y[n], y[n-1] ... y[n-(N-1)]                     
u        = .005;                     %% LMS gain
Estimate = zeros(1,length(y));       %% Storage for the Estimates

%% 4. Start the Simulation Loop
 
Error  = zeros(1,length(y));
for n=1:length(y)
    Y(2:N,1)     = Y(1:N-1,1);    %% Here we shift the observation y[n] 
    Y(1,1)       = y(1,n);        %% into the estimator's shift register
    Estimate(1,n)= EC*Y;          %% Here we form the estimate of x[n]
    if(n>Approximate_Delay+3)
        TrainingSymbol    = In(1, n - Approximate_Delay + 2);       
        Error(1,n)        = TrainingSymbol - Estimate(1,n); %% Calculate Error
        [Error(1,n) TrainingSymbol  Estimate(1,n) n];
        EC = EC + 2*u*Error(1,n)*Y';                 %% LMS Coefficient Update 
    end
end


figure(2);
stem(real(Estimate), 'k'); title('Real Part of Equalizer Output');
axis([600 680 -1.2 1.2]);grid on;

figure(3);
subplot(2,2,1:2);
plot(1:3:700, abs(Error(1,1:3:700)),'k'); grid on; title('The Absolute Value of Error[n]');
xlabel('LMS Iterations');
axis([0 700 0 .6]);grid on;
subplot(2,2,3);
plot(real(y(1,10:end)), imag(y(1,10:end)), 'k.'); grid on; title('Observed Constellation');
xlabel('I'); ylabel('Q');
set(gca, 'XTick', [-1 0 1]);
set(gca, 'YTick', [-1 0 1]);
subplot(2,2,4);
plot(real(Estimate(1,200:end)), imag(Estimate(1,200:end)), 'k.'); grid on;
title('Equalized Constellation after 200 LMS Updates');
xlabel('I'); ylabel('Q');
set(gca, 'XTick', [-1 0 1]);
set(gca, 'YTick', [-1 0 1]);
