%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 4 - Section 5: DFE1 - Decision Feedback Equalizer Example
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger                          Date: May 2013

%  This example shows the impact of removing channel post cursors
%  by programming only the feedback portion of the DFE. The Feed
%  forward portion is programmed as a pass through element.


clc;
clear all;
close all;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Pages 245-246)

%% 0. Setting up the QPSK source signal and Distortion Channel
Length   = 200;
QPSK     = sign(randn(1,Length)) + j*sign(randn(1,Length));
Channel  = [0.05  0.1*j  1.0  0.20  -0.15*j  0.1];
y        = filter(Channel,1,QPSK); %% The observation vector
 
%% 1. The Feed Forward Coefficients set up as a Pass Through Element
FF_Coeff = [0 0 0 1 0];            %% Pass Through Coefficients
FF_Out   = filter(FF_Coeff, 1, y); %% Result of Feed Forward FIR

%% 2. Decision Feedback Loop
FB_Pipe = zeros(1,3);              %% Feedback Pipeline
FB      = [-0.2 0.15*j -0.1];      %% Feedback coefficients
Estimate= zeros(1,Length);
LastEstimate = 0;

for n = 1:Length
    Decision       = sign(real(LastEstimate))+j*sign(imag(LastEstimate));
    FB_Pipe(1,2:3) = FB_Pipe(1,1:2);
    FB_Pipe(1,1)   = Decision;
    Estimate(1,n)  = FF_Out(1,n) + FB_Pipe*FB.';
    LastEstimate   = Estimate(1,n);
end

figure(1);
subplot(1,2,1);
plot(imag(y(1,7:end)), real(y(1,7:end)), 'ko', 'MarkerSize', 3, 'LineWidth', .8); grid on;
set(gca, 'XTick', [-1 0 1]);
set(gca, 'YTick', [-1 0 1]);
axis([-1.5 1.5 -1.5 1.5]); title('Observation y[n]'); xlabel('I'); ylabel('Q');
set(gca,  'LineWidth', 1);
subplot(1,2,2);
plot(imag(Estimate(1,9:end)), real(Estimate(1,9:end)), 'ko', 'MarkerSize', 3, 'LineWidth', .8); grid on;
set(gca, 'XTick', [-1 0 1]);
set(gca, 'YTick', [-1 0 1]);
axis([-1.5 1.5 -1.5 1.5]);title('Estimate of x[n]'); xlabel('I'); ylabel('Q');
set(gca,  'LineWidth', 1);

%% 3. EVM performance evaluation
%     At the point of observation y[n]
Ideal = sign(real(y(1,5:end))) + j*sign(imag(y(1,5:end)));
Error = y(1,5:end) - Ideal;
EVM_Observation = 10*log10(var(Error)/var(Ideal))

%     Of the estimate 
Ideal = sign(real(Estimate(1,5:end))) + j*sign(imag(Estimate(1,5:end)));
Error = Estimate(1,15:end) - Ideal(1,11:end);
EVM_Estimate   = 10*log10(var(Error)/var(Ideal))

