%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 4 - Section 5: DFE2 - Decision Feedback Equalizer Example
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger                          Date: May 2013

%  We now upgrade the DFE1 example by calculating the MMSE optimal
%  feed forward coefficient to remove the precursor effects.

clc;
clear all;
close all;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Page 247-248)

%% 0. Setting up the QPSK source signal and Distortion Channel
Length   = 200;
QPSK     = sign(randn(1,Length)) + j*sign(randn(1,Length));
h        = [0.05  0.1*j  1.0  0.20  -0.15*j  0.1];
y        = filter(h,1,QPSK);   %% The observation vector
 
%% 1. The MMSE Optimal Feed Forward Coefficient Calculation
%  1a. Setup information
VarIn   = 1;                  %% Variance of input signal
VarN    = 0;                  %% Noise Variance
Channel = [.05 .1*j  1];
N       = 5;                  %% Number of taps in Feed Forward FIR
L       = length(Channel);    %% Number of taps in channel
D       = N + L - 2;

%% 1b. Calculate H
row = zeros(1,N+L-1); row(1,1:L) = Channel;
col = zeros(1,N);     col(1,1)   = Channel(1,1);
H   = toeplitz(col,row);

%% 1c. Calculate RIn, Rv, Ry
RIn = VarIn*eye(N+L-1);
Rv  = VarN*eye(N);
Ry  = H*RIn*H' + Rv;

%% 1d. Calculate H E(In(n-D))IN(n)  
E_InIN          = zeros(1, N+L-1);
E_InIN(1, D+1)  = VarIn;               
E_InIN_H        = E_InIN*H';
EC              = E_InIN_H*inv(Ry)   %% Column Vector EC = C'

FF_Out          = filter(EC, 1, y);   %% Result of Feed Forward FIR

%% 2. Decision Feedback Loop
FB_Pipe = zeros(1,3);                 %% Feedback Pipeline
FB      = [-0.2 0.15*j -0.1];         %% Feedback coefficients
Estimate= zeros(1,Length);
LastEstimate = 0;

for n = 1:Length
    Decision       = sign(real(LastEstimate))+j*sign(imag(LastEstimate));
    FB_Pipe(1,2:3) = FB_Pipe(1,1:2);
    FB_Pipe(1,1)   = Decision;
    Estimate(1,n)  = FF_Out(1,n) + FB_Pipe*FB.';
    LastEstimate   = Estimate(1,n);
end

figure(1);
subplot(1,2,1);
plot(imag(y(1,7:end)), real(y(1,7:end)), 'ko', 'MarkerSize', 3, 'LineWidth', .8); grid on;
set(gca, 'XTick', [-1 0 1]);
set(gca, 'YTick', [-1 0 1]);
axis([-1.5 1.5 -1.5 1.5]); title('Observation y[n]'); xlabel('I'); ylabel('Q');
set(gca,  'LineWidth', 1);
subplot(1,2,2);
plot(imag(Estimate(1,9:end)), real(Estimate(1,9:end)), 'ko', 'MarkerSize', 3, 'LineWidth', .8); grid on;
set(gca, 'XTick', [-1 0 1]);
set(gca, 'YTick', [-1 0 1]);
axis([-1.5 1.5 -1.5 1.5]);title('Estimate of x[n]'); xlabel('I'); ylabel('Q');
set(gca,  'LineWidth', 1);

%% 3. EVM performance evaluation
%     At the point of observation y[n]
Ideal = sign(real(y(1,5:end))) + j*sign(imag(y(1,5:end)));
Error = y(1,5:end) - Ideal;
EVM_Observation = 10*log10(var(Error)/var(Ideal))

%     Of the estimate 
Ideal = sign(real(Estimate(1,5:end))) + j*sign(imag(Estimate(1,5:end)));
Error = Estimate(1,10:end) - Ideal(1,6:end);
EVM_Estimate   = 10*log10(var(Error)/var(Ideal))
