%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 4 - Section 5: DFE3 - Decision Feedback Equalizer Example
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger                          Date: May 2013


clc;
clear all;
close all;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Page 249-250)

%% 0. Setting up the QPSK source signal and Distortion Channel
Length   = 1000;
QPSK     = sign(randn(1,Length)) + j*sign(randn(1,Length));
h        = [0.05  0.1*j  1.0  0.20  -0.15*j  0.1];
y        = filter(h,1,QPSK);   %% The observation vector

%% 1. Initialize the 5 tap feed forward section 
EC  = [.004 .0109j -0.06 -0.1j 1.0]; %% Initialize with MMSE
                                     %% Optimal coefficients                 
[M GroupDealy_Channel]   = max(h);
[M GroupDelay_Estimator] = max(EC);
TotalGroupDelay          = GroupDealy_Channel + GroupDelay_Estimator;
N = length(EC);              %% Number of taps Feed Forward Section
Y = zeros(N,1);              %% Y -> content of estimator shift register
                             %% Y = y[n], y[n-1] ... y[n-(N-1)          
u        = .005;             %% LMS gain
FF_Out         = zeros(1,length(y));  %% Output of Feed Forward Section
Estimate       = zeros(1,length(y));  %% The Estimate of In[n]

%% 2. Start the Adaptive Loop
FB_ShiftReg = zeros(3,1);          %% Feedback shift reg as column vector
DC          = [-0.2 0.15*j -0.1];  %% Feedback coefficients initialized 
LastEstimate = 0;
Error  = zeros(1,length(y));
for n=1:length(y)
    Y(2:N,1)     = Y(1:N-1,1);    %% Here we shift the observation y[n] 
    Y(1,1)       = y(1,n);        %% into the estimator's shift register
    FF_Out(1,n)  = EC*Y;          %% Output of feed forward FIR
    Decision     = sign(real(LastEstimate)) + j *sign(imag(LastEstimate));
    
    FB_ShiftReg(2:3,1) = FB_ShiftReg(1:2,1); %% Shifting Decision into 
    FB_ShiftReg(1,1)   = Decision;           %% Feedback shift register
    Estimate(1,n)      = FF_Out(1,n) + DC*FB_ShiftReg; %% Summing Node
    LastEstimate       = Estimate(1,n);
    if(n>TotalGroupDelay+3)
        TrainingSymbol= QPSK(1, n-TotalGroupDelay+2);       
        Error(1,n)    = TrainingSymbol - Estimate(1,n); %% Calculate Error

        EC = EC + 2*u*Error(1,n)*Y';  %% LMS Coefficient Update Feed Forward
        DC = DC + 2*u*Error(1,n)*FB_ShiftReg'; 
                                    %% LMS Coefficient Update Feedback
    end
end
DC
EC

figure(1);
subplot(1,2,1);
plot(imag(y(1,600:end)), real(y(1,600:end)), 'ko', 'MarkerSize', 3, 'LineWidth', .8); grid on;
set(gca, 'XTick', [-1 0 1]);
set(gca, 'YTick', [-1 0 1]);
axis([-1.5 1.5 -1.5 1.5]); title('Observation y[n]'); xlabel('I'); ylabel('Q');
set(gca,  'LineWidth', 1);
subplot(1,2,2);
plot(imag(Estimate(1,600:end)), real(Estimate(1,600:end)), 'ko', 'MarkerSize', 2, 'LineWidth', .8); grid on;
set(gca, 'XTick', [-1 0 1]);
set(gca, 'YTick', [-1 0 1]);
axis([-1.5 1.5 -1.5 1.5]);title('Estimate of x[n] (LMS FF/FB)'); xlabel('I'); ylabel('Q');
set(gca,  'LineWidth', 1);

figure(2);
plot(abs(Error), 'k'); grid on;

%% 3. EVM performance evaluation
%     At the point of observation y[n]
Ideal = sign(real(y(1,600:end))) + j*sign(imag(y(1,600:end)));
Error = y(1,600:end) - Ideal;
EVM_Observation = 10*log10(var(Error)/var(Ideal))

%     Of the estimate 
Ideal = sign(real(Estimate(1,600:end))) + j*sign(imag(Estimate(1,600:end)));
Error = Estimate(1,600:end) - Ideal;
EVM_Estimate   = 10*log10(var(Error)/var(Ideal))
