%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 5 - Section 3: DC Offset Simulation
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger                              May 2013

clc;
clear all;
close all;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Page 304-305)

rand('seed', 0);

%% Discrete Setup Information
SampleRate = 10e6;          %% 10MHz
Ts         = 1/SampleRate;  %% Sample Period

%% Discrete Gain Information
A_100KHz   = 0.0606;       %% Feedback gain for 100KHz Frequency Corner
A_10KHz    = 0.00623;      %% Feedback gain for 10KHz Frequency Corner
A_1KHz     = 0.000625;     %% Feedback gain for 1KHz Frequency Corner

D_100KHz   = -0.9697;      %% Feedforward gain for 100KHz Frequency Corner
D_10KHz    = -0.9969;      %% Feedforward gain for 10KHz Frequency Corner
D_1KHz     = -0.9997;      %% Feedforward gain for 1KHz Frequency Corner

f          = -0.5*SampleRate:.001*SampleRate:0.5*SampleRate;
z          = exp(j*2*pi*f/SampleRate);
F_100KHz   = D_100KHz*(z-1)./(z-1+A_100KHz);
F_10KHz    = D_10KHz*(z-1)./(z-1+A_10KHz);
F_1KHz     = D_1KHz*(z-1)./(z-1+A_1KHz);

%% BPSK Signal Generation
ModulationFormat = 1;    % 0/1/2/3  =  BPSK/QPSK/8PSK/16QAM
NumberOfSymbols  = 30000;
Alpha            = .5;   % This is the raised cosine index
Delay            = 4;    % Recommended -> Delay = 3 for Alpha > .3
                         % Recommended -> Delay = 6 for .1 > Alpha > .3
                         % Recommended -> Delay > 6 for Alpha < .1
OSR              = 4;    % Recommended --> 4 or 8
InputBits        = round(rand(1, NumberOfSymbols...
                          *(ModulationFormat + 1)));

[OutputSymbol]   = Mapper(InputBits, ModulationFormat);
[BPSK_Signal]    = TX_Waveform_Mapping([OutputSymbol ], OSR, Alpha, Delay);
     
[Mag f1] = Spectrum_Fast(BPSK_Signal, 256, SampleRate);



%% Time, BPSK amplitude and VGA gain Definitions
TimeStart      = 4000;
TimeThreshold1 = 4150;
TimeThreshold2 = 4400;
BPSK_Amplitude = [.02*ones(1,4000) 2*ones(1,1e6)]; 
VGA_Gain       = [28*ones(1,4000)  28:-.2:.4   .3*ones(1,1e6)]; 
                    
VGA_In          = zeros(1,length(BPSK_Signal)); %% For Data Logging
VGA_Out         = zeros(1,length(BPSK_Signal)); %% For Data Logging
DC_Block2_In    = zeros(1,length(BPSK_Signal)); %% For Data Logging
AccumulatorOut  = zeros(1,length(BPSK_Signal)); %% For Data Logging
DC_Block2_Out   = zeros(1,length(BPSK_Signal)); %% For Data Logging

DC_Offset3 = 0.01;     %% Defining DC Offset 3
DC_Offset4 = -.02;     %% Defining DC Offset 4

LastAccumulatorOut = 0;
for i = 2:length(BPSK_Signal)
    if(i > TimeStart && i < TimeThreshold1)
        A = A_100KHz;
        D = D_100KHz;
    elseif(i > TimeThreshold1 && i < TimeThreshold2)
        A = A_10KHz;
        D = D_10KHz;
    else
        A = A_1KHz;
        D = D_1KHz;
    end
    
    VGA_In(1,i)          = real(BPSK_Signal(1,i))*BPSK_Amplitude(1,i) + DC_Offset3;
    VGA_Out(1,i)         = VGA_In(1,i) * VGA_Gain(1,i);
    if(VGA_Out(1,i) > 2);  VGA_Out(1,i) = 2;  end; %% Clipping
    if(VGA_Out(1,i) < -2); VGA_Out(1,i) = -2; end; %% Clipping
    
    DC_Block2_In(1,i)  = VGA_Out(1,i) + DC_Offset4; 
    
    %% Here is the high pass filter routine
    DC_Block2_Out(1,i) = DC_Block2_In(1,i)  * D - LastAccumulatorOut;
    NewAccumulatorOut  = DC_Block2_Out(1,i) * A + LastAccumulatorOut;
    LastAccumulatorOut = NewAccumulatorOut; %% Delay Element
    AccumulatorOut(1,i)= NewAccumulatorOut; %% For later plotting
end
    
figure(10);
subplot(2,1,1);
plot(real(DC_Block2_Out), 'k'); grid on; title('Output of DC Block 2 Module');
axis([3800 4800 -1.2 1.2]); set(gca, 'YTick', [-1 -.5 .5 1]);
subplot(2,1,2);
plot(AccumulatorOut, 'k');  title('Accumulator Output'); hold on;
plot([TimeStart TimeStart+1], [-1 1], 'k:');
plot([TimeThreshold1 TimeThreshold1 + 1], [-1 1], 'k:');
plot([TimeThreshold2 TimeThreshold2 + 1], [-1 1], 'k:');
axis([3800 4800 -.8 .8]);


return
figure(3);
plot(VGA_Gain,'k'); hold on;
plot(BPSK_Amplitude, 'k:'); 
legend('VGA Gain', 'BPSK Peak Amplitude');
axis([3800 4800 -1 30]);

return
figure(2);
subplot(3,1,1);
plot(0.5*real(BPSK_Signal), 'k'); grid on; title('In-Phase Portion of BPSK Signal');
axis([2800 3000 -.6 .6]);
subplot(3,1,2);
plot(f1, Mag - max(Mag), 'k'); grid on; title('BPSK Signal Spectrum');
xlabel('Frequency in Hz'); ylabel('dB');
axis([-0.5*SampleRate 0.5*SampleRate -60 0]);
set(gca, 'XTick', [-4e6 -2e6 0 2e6 4e6]);
set(gca, 'YTick', [-60 -40 -20 0]);

subplot(3,1,3);
plot(f, abs(F_100KHz), 'k'); hold on; grid on;
plot(f, abs(F_10KHz), 'k:'); 
title('High Pass Filter Magnitude Response'); 
axis([-0.5*SampleRate 0.5*SampleRate 0 1.2]);
xlabel('Frequency in Hz');
set(gca, 'XTick', [-4e6 -2e6 0 2e6 4e6]);
set(gca, 'YTick', [0 1]);
legend('100Khz Corner', '10KHz Corner');