%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 5 - Section 3:  Phase Noise Generator function
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger                              May 2013
%% 
%% Out --> is the sequence Theta(t)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Page 282-283)
 
function  [Out Pow_Pos Freq_Pos RMS_PE] = ...
          PhaseNoiseGenerator(SampleRate, NumberOfSamples, dBc, Freq)
 
%% Step 1. Sample the Phase Noise Profile ...
%     (Don't include 0Hz or frequencies > 2e6 in the Freq input vector)
 
MaxFreq          = Freq(1,length(Freq));
FreqIndex        = [0 Freq MaxFreq+1 2.1e6];  %% Conditioned frequency index
dBc_Power        = [-200 dBc -200 -200];      %% Conditioned Profile
 
%% 1a. Because the Phase Noise profile's frequency axis is shown as 
%      log format, we linearly interpolate at dB frequencies as shown below.
 
FreqIndex_dB           = 20*log10(FreqIndex+1);
NewFrequencies_dB      = 60:126;  %% 20*log10(1e3Hz) to about 20*log10(2e6Hz)
New_dBc_Power_dB_Freq  = interp1(FreqIndex_dB, dBc_Power, ...
                                             NewFrequencies_dB, 'linear');

%% 1b. Now we interpolate linearly to get a small frequency spacing of 1KHz
NewFrequencies = 0:1e3:2e6;
New_dBc_Power  = interp1(10.^((NewFrequencies_dB)./20),  ...
                 New_dBc_Power_dB_Freq, NewFrequencies,'linear','extrap');

%New_dBc_Power  = interp1(FreqIndex, dBc_Power, NewFrequencies, 'linear');
NewLinearPower = 10.^(New_dBc_Power/10);

%figure(1);
%semilogx(NewFrequencies, New_dBc_Power, 'k'); grid on;


%% 1c. While we are at it, let's find the rms phase error
SSB_Power = 0;   %% Find rms phase error via trapeziodal integration
for i=1:length(NewFrequencies)-1
    Rise = 0.5*NewLinearPower(1,i) + 0.5*NewLinearPower(1,i+1);
    Run  = NewFrequencies(1,i+1) - NewFrequencies(1,i);
    SSB_Power = SSB_Power + Rise*Run;
end
RMS_PE = sqrt(2*SSB_Power);  %% RMS Phase error in radians  




%% Steps 2 and 3. Converting power to magnitude in a format for the IFFT
                                  %% FIR Filter has 1000 Taps
Freq_Pos = (0:999)*4e6/2000;      %% Set up Frequencies for IFFT 
Freq_Neg = (-1000:1:-1)*4e6/2000;  
Pow_Pos  = interp1(NewFrequencies, NewLinearPower, Freq_Pos,'linear');
Pow_Neg  = interp1(NewFrequencies, NewLinearPower, abs(Freq_Neg),'linear');

%% Preparing for IFFT
LinearPower     = [Pow_Pos Pow_Neg];
Magnitude       = sqrt(LinearPower);  %% Set up Gains for IFFT
Temp            = ifft(Magnitude);

%figure(2)
%semilogx(Freq_Pos, 20*log10(sqrt(Pow_Pos)), 'k'); grid on;


%% Step 4: Compute the FIR filter taps
FIR_Taps        = [Temp(1,1001:2000) Temp(1,1:1000)]; 
H = hann(2002);
H1 = H(2:2001,1)';
FIR_Taps        = FIR_Taps.*H1;

%[f Mag_Response Mag_Response_dB Phase_Response Frequency_Response] = ...
%                       FIR_Frequency_Response(FIR_Taps, 4e6);

%figure(3);
%semilogx(f(1,501:1000), Mag_Response_dB(1,501:1000), 'k'); grid on; title('Filter Magnitude Response');
%axis([0 2e6 -130 -80]);

%% 4. Pass White Gaussian Noise Through FIR filter
 
SampleRatio   = SampleRate/4e6;
SamplesAt4MHz = ceil(NumberOfSamples/SampleRatio); 
WGN           = randn(1,2000 + SamplesAt4MHz);  %% Generate WGN

FilterOutput  = filter(FIR_Taps,1,WGN); 
Out4MHz       = FilterOutput(1,1000:end);

%[Mag f] = Spectrum_Fast(Out4MHz, 1024, 4e6);

%figure(7);
%semilogx(f, 30+Mag, 'k'); grid on; %title('Power Spectral Density of Carrier');
%axis([0 2e6 -130 -60]);
 
%% 5. Upsample from 2MHz to the desired Sample Rate
 
SampleTime4MHz = 1:length(Out4MHz);
NewSampleTime  = (1:NumberOfSamples)*1/SampleRatio;
Output         = interp1(SampleTime4MHz, Out4MHz, NewSampleTime,'spline');
OutTemp        = Output(1,1:NumberOfSamples);
RMSOutput      = sqrt(var(OutTemp));
Out            = OutTemp*RMS_PE/RMSOutput;
