%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 5 - Section 3:  Spectrum_Fast (Faster version of Spectrum Function
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger                              May 2013

%  This function will produce an average Magnitude Specturm that is 
%  similar in look and feel to what is seen on the spectrum analyzer

%  Input Arguments
%  1. Input  --> Is the input waveform
%  2. N_DFT  --> The number of desired points for the DFT operation
%  3. SampleRate --> Only useful for the return vector f

%  Output Arguments
%  1. Mag    --> Averaged Magnitude Spectrum in dB
%  2. f      --> Frequency vector associated with Mag


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Unpublished Code

function [Mag f] = Spectrum_Fast(Input, N_DFT, SampleRate)

if(N_DFT < 2)
    msgbox('N_DFT must be larger than 1','Error Message');
    Mag = []; f = [];
    return;
end

if(length(Input) < N_DFT)
    msgbox('Length of Input must be larger than N_DFT','Error Message');
    Mag = []; f = [];
    return;
end

Step  = floor((length(Input)- N_DFT) / 200);
if(Step < 1)
    msgbox('Please increase Input vector length','Error Message');
    Mag = []; f = [];
    return;
end

N             = N_DFT;
m             = 0:1:N-1; % Frequency Index 
Normalized_F  = m/N;     % Normalized Frequencies
Hanning       = hann(N + 2);
H             = (N/sum(Hanning(2:N+1,1)))* Hanning(2:N+1,1)';

DFTOutput     = zeros(1, N);

for Repetitions = 0:199
    n         = (1 + Repetitions*Step):(Repetitions*Step + N);
    h         = Input(1,n);
    DFTOutput = DFTOutput + (1/N)*abs(fft(h.*H));
end

if(mod(N,2) == 0)  %% even number
    NegStart = (N/2) + 1;
    PosStop  = (N/2);
else
    NegStart = ceil(N/2);
    PosStop  = floor(N/2);
end

FFTOutput           = [ DFTOutput(1, NegStart:end) DFTOutput(1,1:PosStop) ];
DiscreteFrequencies = [ Normalized_F(1, NegStart:end)-1  Normalized_F(1,1:PosStop) ];

Mag       = 20*log10(FFTOutput/200);
f         = DiscreteFrequencies*SampleRate;
