%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 6 - Section 3: Total OFDM Simulation
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger                              May 2013


clc;
clear all;
close all;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Page 368-370)

%%                             1. Overall Simulation Setup
%% 1a.Configuring the Transmitter
NumberOf_OFDMSymbols = 1500;
BitsPerSymbol        = 6;              % BitsPerSymbol = 1,2,4,6 =
                                       % BPSK,QPSK,16QAM,64QAM
TransmitterChoice    = 1;              % 0/1 = upsampled / IFFT128

%% 1b. Configuring Defect Model Code
Mode.Multipath             = 0; % 0/1 = exclude/include defect
Mode.ThermalNoise          = 1;
Mode.PhaseNoise            = 0;
Mode.Freq_Offset           = 0;
Mode.IQ_Imbalance          = 0;
Mode.TimingOffset          = 0;
Mode.TimingDrift           = 0;

Settings.SampleRate        = 40e6;        % Hertz
Settings.NumberOfTaps      = 40;          % For Multipath Model
Settings.DelaySpread       = 150e-9;      % Seconds
Settings.SNR_dB            = 35;          % AWG Thermal Noise
Settings.PhaseNoiseProfile = [1e3 10e3 20e3 35e3 50e3 80e3 90e3 100e3 ...
                              120e3 150e3 200e3 300e3 500e3 1e6;  ...
                              -70 -72  -72 -74 -76  -85  -90  -95 ...
                               -100   -105   -110  -120  -130 -140];                               
Settings.FrequencyOffset    = -100e3;     % Hz
Settings.PhaseImbalance     = pi/2000;    % radians
Settings.AmplitudeImbalance_dB = -.1;     % dB
Settings.Sample_Offset      = -1;         % samples at the sample rate  
Settings.Drift_ppm          = -80;        % given above in ppm (parts per
                                          % million

%% 1c. Configuring Miscellaneous Simulation Controls

CorrectFrequencyOffset      = 1;       % 0/1 = no/yes
UseMaxRatio_Combining       = 1;       % 0/1 = don't use / do use
StaticNoise                 = 0;       % 0/1 = different / same noise
NoiseSeed                   = 600;     % any positive integer
SampleAdvance               = 0;       % Min = 0 / Max = 15

if(StaticNoise); randn('state',NoiseSeed); end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%                          2. Transmitter Setup                       %%

if(TransmitterChoice == 1)
    [Sample_Output TX_Symbol_Stream] = ...
            OFDM_TX_802_11_128(NumberOf_OFDMSymbols, BitsPerSymbol); 
else
    [Sample_Output TX_Symbol_Stream] = ...
            OFDM_TX_802_11(NumberOf_OFDMSymbols, BitsPerSymbol);
end                                                  
                                                    
TX_Output_Clean        = [zeros(1,10) Sample_Output zeros(1,10)];

figure(1);
plot(real(TX_Output_Clean), 'r'); grid on; hold on;
plot(imag(TX_Output_Clean), 'b');



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%                             3. Defect Model                         %%

[TX_Output FIR_Taps]   = Defect_Model(TX_Output_Clean, Settings, Mode);  


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%                             4. Receiver Code                        %%

[Corrected_Symbols]   = OFDM_Receiver(TX_Output, ...
                                      SampleAdvance, ...
                                      CorrectFrequencyOffset, ...
                                      NumberOf_OFDMSymbols,...
                                      UseMaxRatio_Combining);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%                         5. Performance Evaluation                  %%

ErrorVectors             = TX_Symbol_Stream - Corrected_Symbols;
Average_ErrorVectorPower = (1/length(ErrorVectors))*(ErrorVectors*ErrorVectors');

EVM = 10*log10(Average_ErrorVectorPower/1);
display(['EVM = ' num2str(EVM) 'dB']);

% EVM Versus Time Calculation
Error_Time = zeros(1, NumberOf_OFDMSymbols);
for i = 1:NumberOf_OFDMSymbols
    Start             = (i-1)*48 +  1;
    Stop              = (i-1)*48 + 48;
    CurrentSymbol     = Corrected_Symbols(1, Start:Stop); 
    IdealSymbol       = TX_Symbol_Stream(1, Start:Stop);
    ErrorVector       = IdealSymbol - CurrentSymbol;
    Error_Time(1,i)   = (1/48)*(ErrorVector*ErrorVector');
end

% EVM Versus Frequency Calculation
Error_Frequency = zeros(1,48);
for i = 1:NumberOf_OFDMSymbols
    Start           = (i-1)*48 +  1;
    Stop            = (i-1)*48 + 48;
    CurrentSymbol   = Corrected_Symbols(1, Start:Stop); 
    IdealSymbol     = TX_Symbol_Stream(1, Start:Stop);
    ErrorVector     = IdealSymbol - CurrentSymbol;
    Error_Frequency = Error_Frequency + ...
                   (ErrorVector.*conj(ErrorVector))/NumberOf_OFDMSymbols; 
end
  
EVM_Frequency       = 10*log10(Error_Frequency);    
EVM_Time            = 10*log10(Error_Time);
PosIndex            = [1:6 8:20 22:26];
NegIndex            = [38:42 44:56 58:63] - 64;

f = -.5:0.001:.5;
Response = zeros(1,length(f));
n = 0:1:length(FIR_Taps)-1;
for d=1:length(f)
    E = exp(j*2*pi*n*f(1,d)/1);
    Response(1,d) = FIR_Taps*E';    
end

MagResponse      = 20*log10(abs(Response));
MagResponse_norm = MagResponse - max(MagResponse);
    
figure(1);
hold off;
subplot(2,2,1);hold off;
plot([PosIndex NegIndex], EVM_Frequency,'k.'); grid on; ylabel('dB')
title('EVM Versus Frequency'); xlabel('Tones'); axis([-27 27 -35 0]);
subplot(2,2,2);hold off;
plot(EVM_Time,'k'); grid on; title('EVM Versus Time');
xlabel('Symbols'); ylabel('dB'); axis([1 130 -35 -10])
subplot(2,2,3); hold off; 
plot(f, MagResponse_norm,'k'); grid on; xlabel('Frequency'); hold on;
title('Magnitude Response of Multipath Filter'); 
axis([-13/64 13/64 -20 5]); ylabel('dB');
subplot(2,2,4);hold off;   
stem(abs(FIR_Taps),'k'); grid on; title('FIR Taps'); xlabel('Symbols'); 
    
figure(2);
plot(real(Corrected_Symbols), imag(Corrected_Symbols),'k.', ...
                                                    'Markersize', 10);
grid on;xlabel('real'); ylabel('imag'); axis([-1.5 1.5 -1.5 1.5]);











