%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 5 - Section 3: Defect Model
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger                              May 2013


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Page 294)

function [Output_Waveform FIR_Taps] = Defect_Model(TX_Samples, ...
                                                   Settings, Mode)    
                                       

%% 1. Transfering the Mode Parameters
Multipath_Select        = Mode.Multipath;  % 0/1  means  exclude/include
AW_GaussianNoise_Select = Mode.ThermalNoise;
PhaseNoise_Select       = Mode.PhaseNoise;
Frequency_Offset_Select = Mode.Freq_Offset;
IQ_Imbalance_Select     = Mode.IQ_Imbalance;
Timing_Offset_Select    = Mode.TimingOffset;
Timing_Drift_Select     = Mode.TimingDrift;
             
%% 2. Transfering Settings
N                       = Settings.NumberOfTaps; % For Multipath Model
DelaySpread             = Settings.DelaySpread;  % 
SampleRate              = Settings.SampleRate;
SNR_dB                  = Settings.SNR_dB;       % Thermal Noise
PhaseNoiseProfile1      = Settings.PhaseNoiseProfile;
FrequencyOffset         = Settings.FrequencyOffset;
PhaseImbalance          = Settings.PhaseImbalance;
AmplitudeImbalance_dB   = Settings.AmplitudeImbalance_dB;
Sample_Offset           = Settings.Sample_Offset;
Drift_ppm               = Settings.Drift_ppm;
  
%% 3. Generating Multipath Filter
FIR_Taps = GetMultipathFilter(SampleRate, DelaySpread, N);
if(Multipath_Select == 1)
    TX_Samples = filter(FIR_Taps, 1, TX_Samples);
    VarOutput  = var(TX_Samples);
    TX_Samples = TX_Samples./sqrt(VarOutput);
end

%% 4. Generating Additive White Gaussian Noise (Thermal Noise)
if(AW_GaussianNoise_Select == 1)
    TX_Samples   = TX_Samples + Generate_AWGN(TX_Samples, SNR_dB);
end

%% 5. Adding Phase Noise
NumberOfSamples      = length(TX_Samples);                   
[PhNoise Gain Freq RMSN] = PhaseNoiseGenerator(SampleRate, ...
    NumberOfSamples, PhaseNoiseProfile1(2,:), PhaseNoiseProfile1(1,:));
PhNoise                  = PhNoise - mean(PhNoise);
PhaseNoise               = exp(j*PhNoise); 

if(PhaseNoise_Select == 1);
    display(['Integrated Phase Noise1: ' num2str(RMSN*57.3)]);
    TX_Samples           = TX_Samples.*PhaseNoise;
end

%% 6. Adding Frequency Offset
OffsetSignal = ...
      exp(j*2*pi*(1:length(TX_Samples))*FrequencyOffset/SampleRate);
if(Frequency_Offset_Select == 1)
    TX_Samples   = TX_Samples.*OffsetSignal;
end

%% 7. Adding IQ Imbalance
IGain                 = 10^(0.5*AmplitudeImbalance_dB/20);
QGain                 = 10^(-0.5*AmplitudeImbalance_dB/20);
 
if(IQ_Imbalance_Select == 1)
    TX_Samples  = IQImbalance(TX_Samples, PhaseImbalance,  IGain, QGain);
end
 
%% 8. Add Timing Offset
if(Timing_Offset_Select == 1)
    TX_Samples     = Cause_TimingOffset(TX_Samples, Sample_Offset);
end
 
%% 9. Add Timing Drift
if(Timing_Drift_Select == 1)
    TX_Samples  = Cause_TimingDrift(TX_Samples, Drift_ppm);
end

Output_Waveform = TX_Samples;
