%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 5 - Section 2: Mapper Function
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger  


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Page 263)

function [OutputSymbols] = Mapper(InputBits, ModulationFormat)
% Modulation format: 0, 1, 2, 3 --> BPSK, QPSK, 8PSK, 16QAM

persistent BPSK_LUT QPSK_LUT PSK8_LUT QAM16_LUT
if(isempty(BPSK_LUT))
   C         = 1/sqrt(2);
   BPSK_LUT  = 1/sqrt(2)* [-1 - j; 1 + j];                           %% [0;1] 
   QPSK_LUT  = 1/sqrt(2)* [-1 - j; -1 + j;  1 - j;  1 + j];          %% [00;01;10;11]
   PSK8_LUT  =            [-C + j*C; -1     ;        j; C + j*C; ... 
                           -j      ; -C - j*C; C - j*C;      1];     %% [000;001;010;011; 100;101;110;111] 
   QAM16_LUT = 1/sqrt(10.6)*[-3 - 3*j; -3 - j; -3 + j; -3 + 3*j; ...  
                             -1 - 3*j; -1 - j; -1 + j; -1 + 3*j; ...   
                              1 - 3*j;  1 - j;  1 + j;  1 + 3*j; ...   
                              3 - 3*j;  3 - j;  3 + j;  3 + 3*j];     %%  [0000;0001;0010;0011; 0100;0101;0110;0111; 1000;1001;1010;1011; 1100;1101;1110;1111]
end

NumberOfSymbols = length(InputBits)/(ModulationFormat + 1);
OutputSymbols   = zeros(1,NumberOfSymbols);

for i = 1:NumberOfSymbols
    Start = 1 + (i - 1)*(ModulationFormat + 1);
    Stop  = Start + ModulationFormat;
    Bit_Group = InputBits(1,Start:Stop);
    
    switch(ModulationFormat)
        case 0
            Code   = Bit_Group(1,1) + 1;
            Symbol = BPSK_LUT(Code, 1);
        case 1
            Code   = Bit_Group(1,2)*2 + Bit_Group(1,1) + 1;
            Symbol = QPSK_LUT(Code, 1);  
        case 2
            Code   = Bit_Group(1,3)*4 + Bit_Group(1,2)*2 ...
                   + Bit_Group(1,1) + 1;
            Symbol = PSK8_LUT(Code, 1);   
        case 3
            Code   = Bit_Group(1,4)*8 + Bit_Group(1,3)*4 ...
                   + Bit_Group(1,2)*2 + Bit_Group(1,1) + 1;
            Symbol = QAM16_LUT(Code, 1);
    end
    OutputSymbols(1,i) = Symbol;
end

