%% Text: Digital Signal Processing in Modern Communication Systems
%% Chapter 6 - Section 3: OFDM_Receiver Functions
%% Copyright: Andreas Schwarzinger May 2013
%%
%% Author: Andreas Schwarzinger                              May 2013

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Code (Page 371-374)

function [Corrected_Symbols] = OFDM_Receiver(TX_Output, ...
                                             SampleAdvance, ...
                                             CorrectFrequencyOffset, ...
                                             NumberOf_OFDMSymbols,...
                                             UseMaxRatio_Combining)

%% 1. The Receiver Code : Decimation by 2
RX_Waveform_20MHz = TX_Output(1,1:2:end);
    

%% 2. Packet Detector  
 
[Comparison_Result PacketDetFlag FallingEdgePosition AutoCorr_Est] ...
       = Packet_Detector(RX_Waveform_20MHz); 

display(['Falling Edge Position: ' num2str(FallingEdgePosition)]);
if(FallingEdgePosition > 600)
    display(['Error in Falling Edge Position: ' 
        num2str(FallingEdgePosition)]);
    return;
end


%% 3. Detect and Correcting the Coarse Frequency Offset Estimate

[FreqOffset]      = Detect_FrequencyOffsets(RX_Waveform_20MHz, ...
                          FallingEdgePosition);
CoarseOffset      = FreqOffset(1,1);
n                 = 1:length(RX_Waveform_20MHz);
NCO_Signal        = exp(-j*2*pi*n*CoarseOffset/20e6);
if(CorrectFrequencyOffset == 1)
    RX_Waveform_20MHz = RX_Waveform_20MHz.*NCO_Signal;
end

%% 4. Detect and Correcting the Fine Frequency Offset Estimate

[FreqOffset] = Detect_FrequencyOffsets(RX_Waveform_20MHz, ...
                               FallingEdgePosition);
FineOffset        = FreqOffset(1,2);
n                 = 1:length(RX_Waveform_20MHz);
NCO_Signal        = exp(-j*2*pi*n*FineOffset/20e6);
if(CorrectFrequencyOffset == 1)
    RX_Waveform_20MHz = RX_Waveform_20MHz.*NCO_Signal;
end


%% 5. Long Training Symbol Correlator
LongTrainingSequence = Get_LongTrainingSequence(1);
%LongTrainingSymbol   = ...
%         LongTrainingSequence(1,33-SampleAdvance:96-SampleAdvance); 
LongTrainingSymbol   = LongTrainingSequence(1,33:96); 

[LTPeak_Value1 LTPeak_Position OutputLong] = LongSymbol_Correlator( ...
        LongTrainingSymbol, RX_Waveform_20MHz, FallingEdgePosition);
       
LTPeak_Position = LTPeak_Position - SampleAdvance; % Insert time advance



%% 6. Channel Estimate and Equalizer Setup
  % --> Extract both long training symbols from received waveform and average them
First_LongSymbol  = RX_Waveform_20MHz(1, LTPeak_Position  - 64: LTPeak_Position  -  1);
Second_LongSymbol = RX_Waveform_20MHz(1, LTPeak_Position      : LTPeak_Position  + 63);
Averaged_LongTraining_Symbol = First_LongSymbol*.5 + Second_LongSymbol*.5;

  % --> Take its FFT and extract the positive and negative tone orientations
FFT_Of_LongTraining_Symbol = (1/64)*fft(Averaged_LongTraining_Symbol);
RX_Positive_Tones          = FFT_Of_LongTraining_Symbol(1, 2:27);
RX_Negative_Tones          = FFT_Of_LongTraining_Symbol(1, 39:64);

  % --> Establish ideal tone orientations
[Ignore AllTones]    = Get_LongTrainingSequence(1);
Ideal_Positive_Tones = AllTones(1, 34:59);
Ideal_Negative_Tones = AllTones(1,  7:32);

  % --> Channel Estimate and equalizer coefficient calculation
Channel_Estimate_Pos     = RX_Positive_Tones ./ Ideal_Positive_Tones;
Channel_Estimate_Neg     = RX_Negative_Tones ./ Ideal_Negative_Tones;
Channel_Estimate         = [0 Channel_Estimate_Pos zeros(1,11) ...
                              Channel_Estimate_Neg];   
    
Equalizer_Positive       = 1./Channel_Estimate_Pos;
Equalizer_Negative       = 1./Channel_Estimate_Neg;
Equalizer_Coefficients   = [0 Equalizer_Positive zeros(1,11) ...
                              Equalizer_Negative];
                         
    
%% 7. Examine Pilot Tones for subsequent Maximum Ratio Combining
    
Pilot_Minus21_Strength   = abs(Channel_Estimate(1, 1 + 43) );
Pilot_Minus07_Strength   = abs(Channel_Estimate(1, 1 + 57) );
Pilot_Plus_07_Strength   = abs(Channel_Estimate(1, 1 +  7) );
Pilot_Plus_21_Strength   = abs(Channel_Estimate(1, 1 + 21) );
    
SumIt = Pilot_Minus21_Strength + Pilot_Minus07_Strength + ...
        Pilot_Plus_07_Strength + Pilot_Plus_21_Strength;
    
if(UseMaxRatio_Combining == 0)
    C1    = 1/4;
    C2    = 1/4;
    C3    = 1/4;
    C4    = 1/4;
else
    C1    = Pilot_Minus21_Strength/SumIt;
    C2    = Pilot_Minus07_Strength/SumIt;
    C3    = Pilot_Plus_07_Strength/SumIt;
    C4    = Pilot_Plus_21_Strength/SumIt;
end


%% 8. Process all Received Symbols
L = 8;
AverageSlopeFilter = zeros(1,L);

Corrected_Symbols = zeros(1, 48*NumberOf_OFDMSymbols);
for i = 1:NumberOf_OFDMSymbols
    Start             = (i-1)*80 + LTPeak_Position + 64 + 16;
    Stop              = (i-1)*80 + LTPeak_Position + 127 + 16;
    CurrentOFDMSymbol =  RX_Waveform_20MHz(1, Start: Stop);
   
    CurrentFFTOutput  = (1/64)*fft(CurrentOFDMSymbol);
    EqualizedSymbol   = CurrentFFTOutput .* Equalizer_Coefficients;

   
   % --> Processing the Phase Drift
    Pilot_Minus21     = EqualizedSymbol(1, 1 + 43);
    Pilot_Minus07     = EqualizedSymbol(1, 1 + 57);
    Pilot_Plus_07     = EqualizedSymbol(1, 1 +  7);
    Pilot_Plus_21     = EqualizedSymbol(1, 1 + 21);
   
    Averaged_Pilot    = C1*Pilot_Minus21 + C2*Pilot_Minus07 + ...
                        C3*Pilot_Plus_07 + C4*Pilot_Plus_21;
    
        %[angle(EqualGain) angle(MaxRatio)]*57.3;
    Theta                  = angle(Averaged_Pilot);
    DerotationScalar       = exp(-j*Theta);
    CorrectedSymbol1       = EqualizedSymbol * DerotationScalar;
    Equalizer_Coefficients = Equalizer_Coefficients  * exp(-j*Theta/L);
   
   
   % --> Remove average phase offset from pilots
    Pilot_Minus21     = EqualizedSymbol(1, 1 + 43)*conj(Averaged_Pilot);
    Pilot_Minus07     = EqualizedSymbol(1, 1 + 57)*conj(Averaged_Pilot);
    Pilot_Plus_07     = EqualizedSymbol(1, 1 +  7)*conj(Averaged_Pilot);
    Pilot_Plus_21     = EqualizedSymbol(1, 1 + 21)*conj(Averaged_Pilot);
        
    Slope             = - C1*angle(Pilot_Minus21)/21  ...
                        - C2*angle(Pilot_Minus07)/7  ...
                        + C3*angle(Pilot_Plus_07)/7  ...
                        + C4*angle(Pilot_Plus_21)/21;
   
    AverageSlopeFilter(1,2:L)  = AverageSlopeFilter(1,1:L-1);
    AverageSlopeFilter(1,1)    = Slope;
    AverageSlope               = sum(AverageSlopeFilter)/(L);
                    
    StepPlus          = (  0:31)*AverageSlope;
    StepMinus         = (-32:-1)*AverageSlope;
    AppliedCorrection = [StepPlus StepMinus];
    CorrectedSymbol2  = CorrectedSymbol1 .* exp(-j*AppliedCorrection);   
    Equalizer_Coefficients = Equalizer_Coefficients .* exp(-j*AppliedCorrection/L);
   
    Start                            = (i-1)*48 +  1;
    Stop                             = (i-1)*48 + 48;
    IFFT_Index                       = [1:6 8:20 22:26 38:42 44:56 58:63];
    Corrected_Symbols(1, Start:Stop) = CorrectedSymbol2(1, 1 + IFFT_Index);
end


